import json
from typing import List, Dict, Any
import os

class VisualDataProcessor:
    def __init__(self, model_name: str, dataset_type: str, file_paths: Dict[str, Dict[str, str]]):
        self.model_name = model_name
        self.dataset_type = dataset_type
        self.file_paths = file_paths[dataset_type][model_name]
        self.dataset_names = ['AndroidControl', 'AITZ', 'GUI_Odyssey']

    @staticmethod
    def read_json(path: str) -> Any:
        with open(path, 'r', encoding='utf-8') as file:
            return json.load(file)

    def get_paired_data(self) -> List[tuple]:
        return [
            (
                self.read_json(self.file_paths["AndroidControl_test_data"]),
                self.read_json(self.file_paths["AndroidControl_res_data"])
            ),
            (
                self.read_json(self.file_paths["AITZ_test_data"]),
                self.read_json(self.file_paths["AITZ_res_data"])
            ),
            (
                self.read_json(self.file_paths["GUI_Odyssey_test_data"]),
                self.read_json(self.file_paths["GUI_Odyssey_res_data"])
            ),
        ]

    @staticmethod
    def get_visual_data(test_data: List[Dict], res_data: Dict, dataset_name: str):
        visual_new_data = [
            item for item in res_data['detailed_results']
            if item['action_type'] == 1 and item['is_success']
        ] 
        image_path_list = [item['image_path'][0] for item in visual_new_data]
        step_list = [item['step_id'] for item in visual_new_data]
        mask_test_data = [
            {**item, "dataset_name": dataset_name}
            for item in test_data
            if item['images'][0] in image_path_list and item['step_id'] in step_list
        ]
        
        mask_res_data = [
            {**item, "dataset_name": dataset_name}
            for item in res_data['detailed_results']
            if item['image_path'][0] in image_path_list and item['step_id'] in step_list
        ]
        return mask_test_data, mask_res_data


    def process(self) -> List[Dict]:
        result1 = []
        result2 = []
        for idx, (test_data, res_data) in enumerate(self.get_paired_data()):
            result1.extend(self.get_visual_data(test_data, res_data, self.dataset_names[idx])[0])
            result2.extend(self.get_visual_data(test_data, res_data, self.dataset_names[idx])[1])
        return result1, result2
    
    def saveJson(self, result, output_path):
        with open(output_path, 'w', encoding='utf-8') as file:
            json.dump(result, file, ensure_ascii=False, indent=2)
        print(f"[INFO] Saved {len(result)} items to {output_path}")

    def save(self, output_path1: str, output_path2: str):
        result1, result2 = self.process()
        self.saveJson(result1, output_path1)
        self.saveJson(result2, output_path2)
        


file_paths = {
    "Low":{
       "OS_Atlas-Pro-7B": {
            "AndroidControl_test_data": "/Agent_ScanKit/datasets/json/AndroidControl/low_test_os_atlas.json",
            "AndroidControl_res_data": "/Agent_ScanKit/results/AndroidControl/AndroidControl_low_OS-Atlas-Pro-7B_raw.json",
            "AITZ_test_data": "/Agent_ScanKit/datasets/json/AITZ/all_test_os_atlas.json",
            "AITZ_res_data": "/Agent_ScanKit/results/AITZ/AITZ_all_OS_Atlas-Pro-7B_raw.json",
            "GUI_Odyssey_test_data": "/Agent_ScanKit/datasets/json/GUIOdyssey/low_random_os_atlas.json",
            "GUI_Odyssey_res_data": "/Agent_ScanKit/results/GUIOdyssey/GUIOdyssey_low_OS_Atlas-Pro-7B_raw.json"
        },
        "OS_Atlas-Pro-4B": {
            "AndroidControl_test_data": "/Agent_ScanKit/datasets/json/AndroidControl/low_test_os_atlas.json",
            "AndroidControl_res_data": "/Agent_ScanKit/results/AndroidControl/AndroidControl_low_OS_Atlas_Pro_4B_raw.json",
            "AITZ_test_data": "/Agent_ScanKit/datasets/json/AITZ/all_test_os_atlas.json",
            "AITZ_res_data": "/Agent_ScanKit/results/AITZ/AITZ_all_OS_Atlas-Pro-4B_raw.json",
            "GUI_Odyssey_test_data": "/Agent_ScanKit/datasets/json/GUIOdyssey/low_random_os_atlas.json",
            "GUI_Odyssey_res_data": "/Agent_ScanKit/results/GUIOdyssey/GUIOdyssey_low_OS_Atlas-Pro-4B_raw.json"
        },
        "OS_Genesis-4B-AC":{
            "AndroidControl_test_data": "/Agent_ScanKit/datasets/json/AndroidControl/low_test_os_genesis.json",
            "AndroidControl_res_data": "/Agent_ScanKit/results/AndroidControl/AndroidControl_low_OS_Genesis-4B-AC_raw.json",
            "AITZ_test_data": "/Agent_ScanKit/datasets/json/AITZ/all_test_os_genesis.json",
            "AITZ_res_data": "/Agent_ScanKit/results/AITZ/AITZ_all_OS_Genesis-4B-AC_raw.json",
            "GUI_Odyssey_test_data": "/Agent_ScanKit/datasets/json/GUIOdyssey/low_random_os_genesis.json",
            "GUI_Odyssey_res_data": "/Agent_ScanKit/results/GUIOdyssey/GUIOdyssey_low_OS_Genesis-4B-AC_raw.json"
        },
        "OS_Genesis-8B-AC":{
            "AndroidControl_test_data": "/Agent_ScanKit/datasets/json/AndroidControl/low_test_os_genesis.json",
            "AndroidControl_res_data": "/Agent_ScanKit/results/AndroidControl/AndroidControl_low_OS_Genesis-8B-AC_raw.json",
            "AITZ_test_data": "/Agent_ScanKit/datasets/json/AITZ/all_low_test_os_genesis.json",
            "AITZ_res_data": "/Agent_ScanKit/results/AITZ/AITZ_all_OS_Genesis-8B-AC_raw.json",
            "GUI_Odyssey_test_data": "/Agent_ScanKit/datasets/json/GUIOdyssey/low_random_os_genesis.json",
            "GUI_Odyssey_res_data": "/Agent_ScanKit/results/GUIOdyssey/GUIOdyssey_low_OS_Genesis-8B-AC_raw.json"
        },
        "OS_Genesis-7B-AC":{
            "AndroidControl_test_data": "/Agent_ScanKit/datasets/json/AndroidControl/low_test_os_genesis.json",
            "AndroidControl_res_data": "/Agent_ScanKit/results/AndroidControl/AndroidControl_low_OS_Genesis_7B_AC_raw.json",
            "AITZ_test_data": "/Agent_ScanKit/datasets/json/AITZ/all_low_test_os_genesis.json",
            "AITZ_res_data": "/Agent_ScanKit/results/AITZ/AITZ_all_low_OS_Genesis-7B-AC_raw.json",
            "GUI_Odyssey_test_data": "/Agent_ScanKit/datasets/json/GUIOdyssey/low_random_os_genesis.json",
            "GUI_Odyssey_res_data": "/Agent_ScanKit/results/GUIOdyssey/GUIOdyssey_low_OS_Genesis-7B-AC_raw.json"
        },
        "GUI_Odyssey":{
            "AndroidControl_test_data": "/Agent_ScanKit/datasets/json/AndroidControl/low_test_gui_odyssey.json",
            "AndroidControl_res_data": "/Agent_ScanKit/results/AndroidControl/AndroidControl_low_GUI_Odyssey_raw.json",
            "AITZ_test_data": "/Agent_ScanKit/datasets/json/AITZ/all_low_test_gui_odyssey.json",
            "AITZ_res_data": "/Agent_ScanKit/results/AITZ/AITZ_all_low_GUI_Odyssey_raw.json",
            "GUI_Odyssey_test_data": "/Agent_ScanKit/datasets/json/GUIOdyssey/low_random_gui_odyssey.json",
            "GUI_Odyssey_res_data": "/Agent_ScanKit/results/GUIOdyssey/GUIOdyssey_low_GUI-Odyssey_raw.json"
        },
        "Aguvis-7B":{
            "AndroidControl_test_data": "/Agent_ScanKit/datasets/json/AndroidControl/low_test_aguvis.json",
            "AndroidControl_res_data": "/Agent_ScanKit/results/AndroidControl/AndroidControl_low_Aguvis_raw.json",
            "AITZ_test_data": "/Agent_ScanKit/datasets/json/AITZ/all_low_test_aguvis.json",
            "AITZ_res_data": "/Agent_ScanKit/results/AITZ/AITZ_all_low_Aguvis-7B-720P_raw.json",
            "GUI_Odyssey_test_data": "/Agent_ScanKit/datasets/json/GUIOdyssey/low_random_aguvis.json",
            "GUI_Odyssey_res_data": "/Agent_ScanKit/results/GUIOdyssey/GUIOdyssey_low_Aguvis-7B-720P_raw.json"
        },
        "UI-TARS-2B-SFT":{
            "AndroidControl_test_data": "/Agent_ScanKit/datasets/json/AndroidControl/low_test_ui_tars.json",
            "AndroidControl_res_data": "/Agent_ScanKit/results/AndroidControl/AndroidControl_low_UI-TARS-2B-SFT_raw.json",
            "AITZ_test_data": "/Agent_ScanKit/datasets/json/AITZ/all_low_test_ui_tars.json",
            "AITZ_res_data": "/Agent_ScanKit/results/AITZ/AITZ_all_UI-TARS-2B-SFT_raw.json",
            "GUI_Odyssey_test_data": "/Agent_ScanKit/datasets/json/GUIOdyssey/low_random_ui_tars.json",
            "GUI_Odyssey_res_data": "/Agent_ScanKit/results/GUIOdyssey/GUIOdyssey_low_UI-TARS-2B-SFT_raw.json"
        },
        "UI-TARS-7B-SFT":{
            "AndroidControl_test_data": "/Agent_ScanKit/datasets/json/AndroidControl/low_test_ui_tars.json",
            "AndroidControl_res_data": "/Agent_ScanKit/results/AndroidControl/AndroidControl_low_UI-TARS-7B-SFT_raw.json",
            "AITZ_test_data": "/Agent_ScanKit/datasets/json/AITZ/all_low_test_ui_tars.json",
            "AITZ_res_data": "/Agent_ScanKit/results/AITZ/AITZ_all_low_UI-TARS-7B-SFT_raw.json",
            "GUI_Odyssey_test_data": "/Agent_ScanKit/datasets/json/GUIOdyssey/low_random_ui_tars.json",
            "GUI_Odyssey_res_data": "/Agent_ScanKit/results/GUIOdyssey/GUIOdyssey_low_UI-TARS-7B-SFT_raw.json"
        },
        "UI-TARS-7B-DPO":{
            "AndroidControl_test_data": "/Agent_ScanKit/datasets/json/AndroidControl/low_test_ui_tars.json",
            "AndroidControl_res_data": "/Agent_ScanKit/results/AndroidControl/AndroidControl_low_UI-TARS-7B-DPO_raw.json",
            "AITZ_test_data": "/Agent_ScanKit/datasets/json/AITZ/all_low_test_ui_tars.json",
            "AITZ_res_data": "/Agent_ScanKit/results/AITZ/AITZ_all_UI-TARS-7B-DPO_raw.json",
            "GUI_Odyssey_test_data": "/Agent_ScanKit/datasets/json/GUIOdyssey/low_random_ui_tars.json",
            "GUI_Odyssey_res_data": "/Agent_ScanKit/results/GUIOdyssey/GUIOdyssey_low_UI-TARS-7B-DPO_raw.json"
        },
        "UI-TARS-1.5-7B":{
            "AndroidControl_test_data": "/Agent_ScanKit/datasets/json/AndroidControl/low_test_ui_tars_1.5.json",
            "AndroidControl_res_data": "/Agent_ScanKit/results/AndroidControl/AndroidControl_low_UI-TARS-1.5-7B_raw.json",
            "AITZ_test_data": "/Agent_ScanKit/datasets/json/AITZ/all_low_test_ui_tars_1.5.json",
            "AITZ_res_data": "/Agent_ScanKit/results/AITZ/AITZ_all_low_UI-TARS-1.5-7B_raw.json",
            "GUI_Odyssey_test_data": "/Agent_ScanKit/datasets/json/GUIOdyssey/low_random_ui_tars_1.5.json",
            "GUI_Odyssey_res_data": "/Agent_ScanKit/results/GUIOdyssey/GUIOdyssey_low_UI-TARS-1.5-7B_raw.json"
        },
        "UI-TARS-72B-SFT":{
            "AndroidControl_test_data": "/Agent_ScanKit/datasets/json/AndroidControl/low_test_ui_tars.json",
            "AndroidControl_res_data": "/Agent_ScanKit/results/AndroidControl/AndroidControl_low_UI-TARS-72B-SFT_raw.json",
            "AITZ_test_data": "/Agent_ScanKit/datasets/json/AITZ/all_low_test_ui_tars.json",
            "AITZ_res_data": "/Agent_ScanKit/results/AITZ/AITZ_all_UI-TARS-72B-SFT_raw.json",
            "GUI_Odyssey_test_data": "/Agent_ScanKit/datasets/json/GUIOdyssey/low_random_ui_tars.json",
            "GUI_Odyssey_res_data": "/Agent_ScanKit/results/GUIOdyssey/GUIOdyssey_low_UI-TARS-72B-SFT_raw.json"
        },
        "UI-TARS-72B-DPO":{
            "AndroidControl_test_data": "/Agent_ScanKit/datasets/json/AndroidControl/low_test_ui_tars.json",
            "AndroidControl_res_data": "/Agent_ScanKit/results/AndroidControl/AndroidControl_low_UI-TARS-72B-DPO_raw.json",
            "AITZ_test_data": "/Agent_ScanKit/datasets/json/AITZ/all_low_test_ui_tars.json",
            "AITZ_res_data": "/Agent_ScanKit/results/AITZ/AITZ_all_UI-TARS-72B-DPO_raw.json",
            "GUI_Odyssey_test_data": "/Agent_ScanKit/datasets/json/GUIOdyssey/low_random_ui_tars.json",
            "GUI_Odyssey_res_data": "/Agent_ScanKit/results/GUIOdyssey/GUIOdyssey_low_UI-TARS-72B-DPO_raw.json"
        },
        "GUI-R1-3B":{
            "AndroidControl_test_data": "/Agent_ScanKit/datasets/json/AndroidControl/low_test_gui_r1.json",
            "AndroidControl_res_data": "/Agent_ScanKit/results/AndroidControl/AndroidControl_low_GUI-R1-3B_raw.json",
            "AITZ_test_data": "/Agent_ScanKit/datasets/json/AITZ/all_low_test_gui_r1.json",
            "AITZ_res_data": "/Agent_ScanKit/results/AITZ/AITZ_all_low_GUI-R1-3B_raw.json",
            "GUI_Odyssey_test_data": "/Agent_ScanKit/datasets/json/GUIOdyssey/low_random_gui_r1.json",
            "GUI_Odyssey_res_data": "/Agent_ScanKit/results/GUIOdyssey/GUIOdyssey_low_GUI-R1-3B_raw.json"
        },
        "Agent-CPM":{
            "AndroidControl_test_data": "/Agent_ScanKit/datasets/json/AndroidControl/low_test_agent_cpm.json",
            "AndroidControl_res_data": "/Agent_ScanKit/results/AndroidControl/AndroidControl_low_Agent-CPM_raw.json",
            "AITZ_test_data": "/Agent_ScanKit/datasets/json/AITZ/all_low_test_agent_cpm.json",
            "AITZ_res_data": "/Agent_ScanKit/results/AITZ/AITZ_all_low_Agent-CPM_raw.json",
            "GUI_Odyssey_test_data": "/Agent_ScanKit/datasets/json/GUIOdyssey/low_random_agent_cpm.json",
            "GUI_Odyssey_res_data": "/Agent_ScanKit/results/GUIOdyssey/GUIOdyssey_low_Agent-CPM_raw.json"
        },
        "GUI-Owl-7B":{
            "AndroidControl_test_data": "/Agent_ScanKit/datasets/json/AndroidControl/low_test_gui_owl.json",
            "AndroidControl_res_data": "/Agent_ScanKit/results/AndroidControl/AndroidControl_low_GUI-Owl-7B_raw.json",
            "AITZ_test_data": "/Agent_ScanKit/datasets/json/AITZ/all_low_test_gui_owl.json",
            "AITZ_res_data": "/Agent_ScanKit/results/AITZ/AITZ_all_low_GUI-Owl-7B_raw.json",
            "GUI_Odyssey_test_data": "/Agent_ScanKit/datasets/json/GUIOdyssey/low_random_gui_owl.json",
            "GUI_Odyssey_res_data": "/Agent_ScanKit/results/GUIOdyssey/GUIOdyssey_low_GUI-Owl-7B_raw.json"
        },
    },
    "High":{
        "OS_Atlas-Pro-7B": {
            "AndroidControl_test_data": "/Agent_ScanKit/datasets/json/AndroidControl/high_test_os_atlas.json",
            "AndroidControl_res_data": "/Agent_ScanKit/results/AndroidControl/AndroidControl_high_OS_Atlas_Pro_7B_raw.json",
            "AITZ_test_data": "/Agent_ScanKit/datasets/json/AITZ/all_high_test_os_atlas.json",
            "AITZ_res_data": "/Agent_ScanKit/results/AITZ/AITZ_all_high_OS_Atlas-Pro-7B_raw.json",
            "GUI_Odyssey_test_data": "/Agent_ScanKit/datasets/json/GUIOdyssey/high_random_os_atlas.json",
            "GUI_Odyssey_res_data": "/Agent_ScanKit/results/GUIOdyssey/GUIOdyssey_high_OS_Atlas-Pro-7B_raw.json"
        },
        "OS_Genesis-7B-AC":{
            "AndroidControl_test_data": "/Agent_ScanKit/datasets/json/AndroidControl/high_test_os_genesis.json",
            "AndroidControl_res_data": "/Agent_ScanKit/results/AndroidControl/AndroidControl_high_OS_Genesis-7B-AC_raw.json",
            "AITZ_test_data": "/Agent_ScanKit/datasets/json/AITZ/all_high_test_os_genesis.json",
            "AITZ_res_data": "/Agent_ScanKit/results/AITZ/AITZ_all_high_OS_Genesis-7B-AC_raw.json",
            "GUI_Odyssey_test_data": "/Agent_ScanKit/datasets/json/GUIOdyssey/high_random_os_genesis.json",
            "GUI_Odyssey_res_data": "/Agent_ScanKit/results/GUIOdyssey/GUIOdyssey_high_OS_Genesis-7B-AC_raw.json"
        },
        "UI-TARS-7B-SFT": {
            "AndroidControl_test_data": "/Agent_ScanKit/datasets/json/AndroidControl/high_test_ui_tars.json",
            "AndroidControl_res_data": "/Agent_ScanKit/results/AndroidControl/AndroidControl_high_UI-TARS-7B-SFT_raw.json",
            "AITZ_test_data": "/Agent_ScanKit/datasets/json/AITZ/all_high_test_ui_tars.json",
            "AITZ_res_data": "/Agent_ScanKit/results/AITZ/AITZ_all_high_UI-TARS-7B-SFT_raw.json",
            "GUI_Odyssey_test_data": "/Agent_ScanKit/datasets/json/GUIOdyssey/high_random_ui_tars.json",
            "GUI_Odyssey_res_data": "/Agent_ScanKit/results/GUIOdyssey/GUIOdyssey_high_UI-TARS-7B-SFT_raw.json"
        },
        "Aguvis-7B":{
            "AndroidControl_test_data": "/Agent_ScanKit/datasets/json/AndroidControl/high_test_aguvis.json",
            "AndroidControl_res_data": "/Agent_ScanKit/results/AndroidControl/AndroidControl_high_Aguvis_raw.json",
            "AITZ_test_data": "/Agent_ScanKit/datasets/json/AITZ/all_high_test_aguvis.json",
            "AITZ_res_data": "/Agent_ScanKit/results/AITZ/AITZ_all_high_Aguvis-7B-720P_raw.json",
            "GUI_Odyssey_test_data": "/Agent_ScanKit/datasets/json/GUIOdyssey/high_random_aguvis.json",
            "GUI_Odyssey_res_data": "/Agent_ScanKit/results/GUIOdyssey/GUIOdyssey_high_Aguvis-7B-720P_raw.json"
        },
        "GUI-R1-7B":{
            "AndroidControl_test_data": "/Agent_ScanKit/datasets/json/AndroidControl/low_test_gui_r1.json",
            "AndroidControl_res_data": "/Agent_ScanKit/results/AndroidControl/AndroidControl_high_GUI-R1-7B_raw.json",
            "AITZ_test_data": "/Agent_ScanKit/datasets/json/AITZ/all_low_test_gui_r1.json",
            "AITZ_res_data": "/Agent_ScanKit/results/AITZ/AITZ_all_high_GUI-R1-7B_raw.json",
            "GUI_Odyssey_test_data": "/Agent_ScanKit/datasets/json/GUIOdyssey/low_random_gui_r1.json",
            "GUI_Odyssey_res_data": "/Agent_ScanKit/results/GUIOdyssey/GUIOdyssey_high_GUI-R1-7B_raw.json"
        },
        "UI-TARS-7B-DPO":{
            "AndroidControl_test_data": "/Agent_ScanKit/datasets/json/AndroidControl/high_test_ui_tars.json",
            "AndroidControl_res_data": "/Agent_ScanKit/results/AndroidControl/AndroidControl_high_UI-TARS-7B-DPO_raw.json",
            "AITZ_test_data": "/Agent_ScanKit/datasets/json/AITZ/all_high_test_ui_tars.json",
            "AITZ_res_data": "/Agent_ScanKit/results/AITZ/AITZ_all_high_UI-TARS-7B-DPO_raw.json",
            "GUI_Odyssey_test_data": "/Agent_ScanKit/datasets/json/GUIOdyssey/high_random_ui_tars.json",
            "GUI_Odyssey_res_data": "/Agent_ScanKit/results/GUIOdyssey/GUIOdyssey_high_UI-TARS-7B-DPO_raw.json"
        },
        "UI-TARS-1.5-7B":{
            "AndroidControl_test_data": "/Agent_ScanKit/datasets/json/AndroidControl/high_test_ui_tars_1.5.json",
            "AndroidControl_res_data": "/Agent_ScanKit/results/AndroidControl/AndroidControl_high_UI-TARS-1.5-7B_raw.json",
            "AITZ_test_data": "/Agent_ScanKit/datasets/json/AITZ/all_high_test_ui_tars_1.5.json",
            "AITZ_res_data": "/Agent_ScanKit/results/AITZ/AITZ_all_high_UI-TARS-1.5-7B_raw.json",
            "GUI_Odyssey_test_data": "/Agent_ScanKit/datasets/json/GUIOdyssey/high_random_ui_tars_1.5.json",
            "GUI_Odyssey_res_data": "/Agent_ScanKit/results/GUIOdyssey/GUIOdyssey_high_UI-TARS-1.5-7B_raw.json"
        },
        "Agent-CPM":{
            "AndroidControl_test_data": "/Agent_ScanKit/datasets/json/AndroidControl/high_test_agent_cpm.json",
            "AndroidControl_res_data": "/Agent_ScanKit/results/AndroidControl/AndroidControl_high_Agent-CPM_raw.json",
            "AITZ_test_data": "/Agent_ScanKit/datasets/json/AITZ/all_high_test_agent_cpm.json",
            "AITZ_res_data": "/Agent_ScanKit/results/AITZ/AITZ_all_high_Agent-CPM_raw.json",
            "GUI_Odyssey_test_data": "/Agent_ScanKit/datasets/json/GUIOdyssey/high_random_agent_cpm.json",
            "GUI_Odyssey_res_data": "/Agent_ScanKit/results/GUIOdyssey/GUIOdyssey_high_Agent-CPM_raw.json"
        },


    }
    
}

if __name__ == "__main__":
    model_name = "GUI-Owl"
    dataset_type = 'High'
    processor = VisualDataProcessor(model_name, dataset_type, file_paths)
    processor.save(f"/Agent_ScanKit/datasets/json/visual_mask/{dataset_type.lower()}/{model_name}.json", f"/Agent_ScanKit/datasets/json/visual_mask/{dataset_type.lower()}/{model_name}_raw.json")
